<?php
include '../includes/header.php';

if (!isset($_GET['id'])) {
    header("Location: produk.php");
    exit();
}

$product_id = $_GET['id'];
$pdo = connectDB();

// Get product details
$stmt = $pdo->prepare("
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE p.id = ? AND p.is_active = 1
");
$stmt->execute([$product_id]);
$product = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$product) {
    header("Location: produk.php");
    exit();
}

// Get related products
$stmt = $pdo->prepare("
    SELECT * FROM products 
    WHERE category_id = ? AND id != ? AND is_active = 1 
    ORDER BY RAND() 
    LIMIT 4
");
$stmt->execute([$product['category_id'], $product_id]);
$related_products = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get product reviews
$stmt = $pdo->prepare("
    SELECT f.*, u.full_name 
    FROM feedback f 
    JOIN users u ON f.user_id = u.id 
    WHERE f.product_id = ? AND f.is_approved = 1 
    ORDER BY f.created_at DESC 
    LIMIT 5
");
$stmt->execute([$product_id]);
$reviews = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate average rating
$stmt = $pdo->prepare("
    SELECT AVG(rating) as avg_rating, COUNT(*) as review_count 
    FROM feedback 
    WHERE product_id = ? AND is_approved = 1
");
$stmt->execute([$product_id]);
$rating_info = $stmt->fetch(PDO::FETCH_ASSOC);
$avg_rating = $rating_info['avg_rating'] ? round($rating_info['avg_rating'], 1) : 0;
$review_count = $rating_info['review_count'] ?: 0;

$image = !empty($product['image']) ? '../assets/images/products/' . $product['image'] : '../assets/images/placeholder.jpg';
$is_low_stock = $product['stock'] < 10;
$is_out_of_stock = $product['stock'] == 0;
?>

<div class="container">
    <!-- Breadcrumb -->
    <nav class="breadcrumb">
        <a href="../index.php">Home</a>
        <span class="separator">/</span>
        <a href="produk.php">Produk</a>
        <span class="separator">/</span>
        <a href="produk.php?category=<?php echo $product['category_id']; ?>"><?php echo $product['category_name']; ?></a>
        <span class="separator">/</span>
        <span class="current"><?php echo $product['name']; ?></span>
    </nav>
    
    <div class="product-detail">
        <div class="product-gallery">
            <div class="main-image">
                <img src="<?php echo $image; ?>" alt="<?php echo $product['name']; ?>" id="mainImage">
                <?php if ($product['is_featured']): ?>
                    <div class="featured-badge">Produk Unggulan</div>
                <?php endif; ?>
                <?php if ($is_out_of_stock): ?>
                    <div class="out-of-stock-overlay">
                        <span>Stok Habis</span>
                    </div>
                <?php elseif ($is_low_stock): ?>
                    <div class="low-stock-badge">Stok Terbatas</div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="product-info">
            <h1><?php echo $product['name']; ?></h1>
            
            <!-- Rating -->
            <div class="product-rating">
                <div class="stars">
                    <?php for ($i = 1; $i <= 5; $i++): ?>
                        <span class="star <?php echo $i <= floor($avg_rating) ? 'filled' : ($i <= $avg_rating ? 'half-filled' : ''); ?>">
                            ★
                        </span>
                    <?php endfor; ?>
                    <span class="rating-value">(<?php echo $avg_rating; ?>)</span>
                </div>
                <span class="review-count"><?php echo $review_count; ?> ulasan</span>
            </div>
            
            <div class="product-price">
                <?php echo formatRupiah($product['price']); ?>
            </div>
            
            <div class="product-meta">
                <div class="meta-item">
                    <i class="fas fa-box"></i>
                    <span>Stok: 
                        <strong class="<?php echo $is_low_stock ? 'text-warning' : 'text-success'; ?>">
                            <?php echo $product['stock']; ?> pcs
                        </strong>
                    </span>
                </div>
                <div class="meta-item">
                    <i class="fas fa-weight"></i>
                    <span>Berat: <?php echo $product['weight']; ?> gram</span>
                </div>
                <div class="meta-item">
                    <i class="fas fa-tag"></i>
                    <span>Kategori: <?php echo $product['category_name']; ?></span>
                </div>
                <div class="meta-item">
                    <i class="fas fa-barcode"></i>
                    <span>SKU: LC<?php echo str_pad($product['id'], 4, '0', STR_PAD_LEFT); ?></span>
                </div>
            </div>
            
            <div class="product-description">
                <h3>Deskripsi Produk</h3>
                <p><?php echo nl2br(htmlspecialchars($product['description'])); ?></p>
            </div>
            
            <?php if (!$is_out_of_stock): ?>
            <div class="purchase-section">
                <div class="quantity-selector">
                    <label for="quantity">Jumlah:</label>
                    <div class="quantity-controls">
                        <button type="button" onclick="decreaseQuantity()" class="quantity-btn">
                            <i class="fas fa-minus"></i>
                        </button>
                        <input type="number" id="quantity" name="quantity" value="1" min="1" max="<?php echo $product['stock']; ?>">
                        <button type="button" onclick="increaseQuantity()" class="quantity-btn">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                    <span class="max-stock">Maks: <?php echo $product['stock']; ?> pcs</span>
                </div>
                
                <div class="product-actions">
                    <button onclick="addToCart(<?php echo $product['id']; ?>, getQuantity())" 
                            class="btn btn-primary btn-large add-to-cart-btn">
                        <i class="fas fa-shopping-cart"></i> Tambah ke Keranjang
                    </button>
                    <button onclick="buyNow(<?php echo $product['id']; ?>, getQuantity())" 
                            class="btn btn-secondary btn-large">
                        <i class="fas fa-bolt"></i> Beli Sekarang
                    </button>
                </div>
            </div>
            <?php else: ?>
            <div class="out-of-stock-section">
                <div class="stock-alert">
                    <i class="fas fa-exclamation-triangle"></i>
                    <div>
                        <h4>Stok Sedang Habis</h4>
                        <p>Produk ini saat ini tidak tersedia. Silakan cek kembali nanti.</p>
                    </div>
                </div>
                <button class="btn btn-secondary btn-large" disabled>
                    <i class="fas fa-bell"></i> Beri Tahu Saat Stok Tersedia
                </button>
            </div>
            <?php endif; ?>
            
            <div class="product-features">
                <div class="feature-item">
                    <i class="fas fa-shipping-fast"></i>
                    <span>Gratis Ongkir Min. Rp 200rb</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-shield-alt"></i>
                    <span>Garansi Kualitas</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-undo"></i>
                    <span>Pengembalian 24 Jam</span>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Product Tabs -->
    <div class="product-tabs">
        <div class="tabs-header">
            <button class="tab-btn active" data-tab="description">Deskripsi</button>
            <button class="tab-btn" data-tab="reviews">Ulasan (<?php echo $review_count; ?>)</button>
            <button class="tab-btn" data-tab="shipping">Pengiriman</button>
        </div>
        
        <div class="tabs-content">
            <div class="tab-pane active" id="description">
                <div class="tab-content">
                    <h3>Detail Produk</h3>
                    <p><?php echo nl2br(htmlspecialchars($product['description'])); ?></p>
                    
                    <div class="product-specs">
                        <h4>Spesifikasi:</h4>
                        <ul>
                            <li><strong>Berat:</strong> <?php echo $product['weight']; ?> gram</li>
                            <li><strong>Kategori:</strong> <?php echo $product['category_name']; ?></li>
                            <li><strong>SKU:</strong> LC<?php echo str_pad($product['id'], 4, '0', STR_PAD_LEFT); ?></li>
                            <li><strong>Status Stok:</strong> 
                                <?php if ($is_out_of_stock): ?>
                                    <span class="text-danger">Habis</span>
                                <?php elseif ($is_low_stock): ?>
                                    <span class="text-warning">Terbatas</span>
                                <?php else: ?>
                                    <span class="text-success">Tersedia</span>
                                <?php endif; ?>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <div class="tab-pane" id="reviews">
                <div class="tab-content">
                    <div class="reviews-summary">
                        <div class="avg-rating">
                            <div class="rating-number"><?php echo $avg_rating; ?></div>
                            <div class="rating-stars">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <span class="star <?php echo $i <= floor($avg_rating) ? 'filled' : ''; ?>">★</span>
                                <?php endfor; ?>
                            </div>
                            <div class="rating-text">Berdasarkan <?php echo $review_count; ?> ulasan</div>
                        </div>
                    </div>
                    
                    <div class="reviews-list">
                        <?php if (count($reviews) > 0): ?>
                            <?php foreach ($reviews as $review): ?>
                            <div class="review-item">
                                <div class="review-header">
                                    <div class="reviewer-info">
                                        <strong><?php echo $review['full_name']; ?></strong>
                                        <div class="review-rating">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <span class="star <?php echo $i <= $review['rating'] ? 'filled' : ''; ?>">★</span>
                                            <?php endfor; ?>
                                        </div>
                                    </div>
                                    <span class="review-date">
                                        <?php echo date('d M Y', strtotime($review['created_at'])); ?>
                                    </span>
                                </div>
                                <div class="review-comment">
                                    <p><?php echo nl2br(htmlspecialchars($review['comment'])); ?></p>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="no-reviews">
                                <p>Belum ada ulasan untuk produk ini.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <div class="tab-pane" id="shipping">
                <div class="tab-content">
                    <h3>Informasi Pengiriman</h3>
                    <div class="shipping-info">
                        <div class="shipping-item">
                            <i class="fas fa-shipping-fast"></i>
                            <div>
                                <h4>Pengiriman Cepat</h4>
                                <p>Pesanan diproses dalam 2-4 jam untuk area Balikpapan</p>
                            </div>
                        </div>
                        
                        <div class="shipping-item">
                            <i class="fas fa-money-bill-wave"></i>
                            <div>
                                <h4>Gratis Ongkir</h4>
                                <p>Gratis ongkos kirim untuk pembelian minimal Rp 200.000</p>
                            </div>
                        </div>
                        
                        <div class="shipping-item">
                            <i class="fas fa-truck"></i>
                            <div>
                                <h4>Area Pengiriman</h4>
                                <p>Kami melayani pengiriman ke seluruh area Balikpapan dan sekitarnya</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Related Products -->
    <?php if (count($related_products) > 0): ?>
    <section class="related-products">
        <h2>Produk Serupa</h2>
        <div class="products-grid">
            <?php foreach ($related_products as $related): 
                $related_image = !empty($related['image']) ? '../assets/images/products/' . $related['image'] : '../assets/images/placeholder.jpg';
                $related_is_out_of_stock = $related['stock'] == 0;
            ?>
            <div class="product-card">
                <div class="product-image">
                    <img src="<?php echo $related_image; ?>" alt="<?php echo $related['name']; ?>"
                         onclick="window.location.href='detail.php?id=<?php echo $related['id']; ?>'">
                    <?php if ($related_is_out_of_stock): ?>
                        <div class="out-of-stock-overlay">
                            <span>Stok Habis</span>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="product-info">
                    <h3 onclick="window.location.href='detail.php?id=<?php echo $related['id']; ?>'">
                        <?php echo $related['name']; ?>
                    </h3>
                    <p class="product-price"><?php echo formatRupiah($related['price']); ?></p>
                    <div class="product-actions">
                        <a href="detail.php?id=<?php echo $related['id']; ?>" class="btn btn-outline">Detail</a>
                        <button onclick="addToCart(<?php echo $related['id']; ?>)" class="btn btn-primary"
                                <?php echo $related_is_out_of_stock ? 'disabled' : ''; ?>>
                            <?php echo $related_is_out_of_stock ? 'Stok Habis' : 'Tambah'; ?>
                        </button>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </section>
    <?php endif; ?>
</div>

<script>
function getQuantity() {
    return parseInt(document.getElementById('quantity').value);
}

function increaseQuantity() {
    const input = document.getElementById('quantity');
    const max = parseInt(input.getAttribute('max'));
    if (input.value < max) {
        input.value = parseInt(input.value) + 1;
    } else {
        alert('Stok maksimum adalah ' + max + ' pcs');
    }
}

function decreaseQuantity() {
    const input = document.getElementById('quantity');
    if (input.value > 1) {
        input.value = parseInt(input.value) - 1;
    }
}

function buyNow(productId, quantity) {
    addToCart(productId, quantity);
    setTimeout(() => {
        window.location.href = 'keranjang.php';
    }, 1000);
}

// Tab functionality
document.addEventListener('DOMContentLoaded', function() {
    const tabBtns = document.querySelectorAll('.tab-btn');
    const tabPanes = document.querySelectorAll('.tab-pane');
    
    tabBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const tabId = this.getAttribute('data-tab');
            
            // Remove active class from all buttons and panes
            tabBtns.forEach(b => b.classList.remove('active'));
            tabPanes.forEach(p => p.classList.remove('active'));
            
            // Add active class to current button and pane
            this.classList.add('active');
            document.getElementById(tabId).classList.add('active');
        });
    });
    
    // Add to cart animation
    const addToCartBtn = document.querySelector('.add-to-cart-btn');
    if (addToCartBtn) {
        addToCartBtn.addEventListener('click', function() {
            this.innerHTML = '<i class="fas fa-check"></i> Ditambahkan!';
            this.classList.add('added');
            
            setTimeout(() => {
                this.innerHTML = '<i class="fas fa-shopping-cart"></i> Tambah ke Keranjang';
                this.classList.remove('added');
            }, 2000);
        });
    }
});
</script>

<style>
.breadcrumb {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-bottom: 2rem;
    font-size: 0.9rem;
}

.breadcrumb a {
    color: #74b9ff;
    text-decoration: none;
}

.breadcrumb a:hover {
    text-decoration: underline;
}

.breadcrumb .separator {
    color: #b2bec3;
}

.breadcrumb .current {
    color: #636e72;
    font-weight: 600;
}

.product-detail {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 3rem;
    margin-bottom: 3rem;
}

.product-gallery {
    position: relative;
}

.main-image {
    position: relative;
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

.main-image img {
    width: 100%;
    height: 400px;
    object-fit: cover;
}

.featured-badge {
    position: absolute;
    top: 15px;
    left: 15px;
    background: #ff6b6b;
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
}

.out-of-stock-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.7);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    font-weight: 600;
}

.low-stock-badge {
    position: absolute;
    bottom: 15px;
    right: 15px;
    background: #ffc107;
    color: #212529;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
}

.product-info h1 {
    margin: 0 0 1rem 0;
    color: #2d3436;
    font-size: 2rem;
}

.product-rating {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.stars {
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.star {
    color: #ddd;
    font-size: 1.2rem;
}

.star.filled {
    color: #ffc107;
}

.star.half-filled {
    background: linear-gradient(90deg, #ffc107 50%, #ddd 50%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
}

.rating-value {
    font-weight: 600;
    color: #2d3436;
}

.review-count {
    color: #636e72;
    font-size: 0.9rem;
}

.product-price {
    font-size: 2rem;
    font-weight: 700;
    color: #ff6b6b;
    margin-bottom: 1.5rem;
}

.product-meta {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: #f8f9fa;
    border-radius: 10px;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.meta-item i {
    color: #74b9ff;
    width: 20px;
}

.text-warning {
    color: #e17055;
}

.text-success {
    color: #00b894;
}

.text-danger {
    color: #d63031;
}

.product-description h3 {
    margin: 0 0 1rem 0;
    color: #2d3436;
}

.product-description p {
    color: #636e72;
    line-height: 1.6;
    margin-bottom: 2rem;
}

.purchase-section {
    background: white;
    padding: 2rem;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    margin-bottom: 2rem;
}

.quantity-selector {
    margin-bottom: 1.5rem;
}

.quantity-selector label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #2d3436;
}

.quantity-controls {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
}

.quantity-btn {
    width: 45px;
    height: 45px;
    border: 2px solid #e9ecef;
    background: white;
    border-radius: 5px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1rem;
    transition: all 0.3s;
}

.quantity-btn:hover {
    border-color: #74b9ff;
    color: #74b9ff;
}

.quantity-selector input {
    width: 80px;
    padding: 0.8rem;
    border: 2px solid #e9ecef;
    border-radius: 5px;
    text-align: center;
    font-weight: 600;
    font-size: 1.1rem;
}

.max-stock {
    font-size: 0.9rem;
    color: #636e72;
}

.product-actions {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.add-to-cart-btn.added {
    background: #00b894;
}

.out-of-stock-section {
    background: white;
    padding: 2rem;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    margin-bottom: 2rem;
}

.stock-alert {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #fff3cd;
    border: 1px solid #ffeaa7;
    border-radius: 5px;
}

.stock-alert i {
    color: #e17055;
    font-size: 1.5rem;
}

.stock-alert h4 {
    margin: 0 0 0.3rem 0;
    color: #856404;
}

.stock-alert p {
    margin: 0;
    color: #856404;
    font-size: 0.9rem;
}

.product-features {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.feature-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem;
    background: white;
    border-radius: 5px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
}

.feature-item i {
    color: #74b9ff;
}

/* Tabs */
.product-tabs {
    margin: 3rem 0;
}

.tabs-header {
    display: flex;
    border-bottom: 2px solid #e9ecef;
    margin-bottom: 2rem;
}

.tab-btn {
    padding: 1rem 2rem;
    background: none;
    border: none;
    font-size: 1rem;
    font-weight: 600;
    color: #636e72;
    cursor: pointer;
    border-bottom: 3px solid transparent;
    transition: all 0.3s;
}

.tab-btn.active {
    color: #ff6b6b;
    border-bottom-color: #ff6b6b;
}

.tab-btn:hover:not(.active) {
    color: #2d3436;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

.tab-content {
    background: white;
    padding: 2rem;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.reviews-summary {
    display: flex;
    align-items: center;
    gap: 2rem;
    margin-bottom: 2rem;
    padding-bottom: 2rem;
    border-bottom: 1px solid #e9ecef;
}

.avg-rating {
    text-align: center;
}

.rating-number {
    font-size: 3rem;
    font-weight: 700;
    color: #2d3436;
    margin-bottom: 0.5rem;
}

.rating-text {
    color: #636e72;
    font-size: 0.9rem;
}

.review-item {
    padding: 1.5rem 0;
    border-bottom: 1px solid #f1f3f4;
}

.review-item:last-child {
    border-bottom: none;
}

.review-header {
    display: flex;
    justify-content: between;
    align-items: center;
    margin-bottom: 1rem;
}

.reviewer-info strong {
    color: #2d3436;
}

.review-date {
    color: #636e72;
    font-size: 0.9rem;
}

.review-comment p {
    margin: 0;
    color: #636e72;
    line-height: 1.6;
}

.shipping-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 5px;
}

.shipping-item i {
    color: #74b9ff;
    font-size: 1.5rem;
}

.shipping-item h4 {
    margin: 0 0 0.3rem 0;
    color: #2d3436;
}

.shipping-item p {
    margin: 0;
    color: #636e72;
}

.related-products {
    margin: 4rem 0;
}

.related-products h2 {
    text-align: center;
    margin-bottom: 2rem;
    color: #2d3436;
}

@media (max-width: 768px) {
    .product-detail {
        grid-template-columns: 1fr;
    }
    
    .product-meta {
        grid-template-columns: 1fr;
    }
    
    .product-actions {
        grid-template-columns: 1fr;
    }
    
    .tabs-header {
        flex-direction: column;
    }
    
    .tab-btn {
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }
    
    .reviews-summary {
        flex-direction: column;
        text-align: center;
    }
}
</style>

<?php include '../includes/footer.php'; ?>